<?php

namespace App\Http\Controllers;

use App\Models\StockPotato;
use App\Models\StoneType;
use Illuminate\Http\Request;
use Endroid\QrCode\Builder\Builder;
use Illuminate\Support\Facades\Storage;

class StockPotatoController extends Controller
{
    public function index()
    {
        $stockPotatoes = StockPotato::with('stoneType')->get();
        return view('stock_potatoes.index', compact('stockPotatoes'));
    }

    public function create()
    {
        $stoneTypes = StoneType::all();
        return view('stock_potatoes.add', compact('stoneTypes'));
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'stone_type_id' => 'required|exists:stone_types,id',
            'length' => 'required|numeric',
            'width' => 'required|numeric',
            'height' => 'required|numeric',
            'quantity' => 'required|integer',
        ]);

        $request->validate([
            'stone_type_id' => 'required|exists:stone_types,id',
            'length' => 'required|numeric',
            'width' => 'required|numeric',
            'height' => 'required|numeric',
            'quantity' => 'required|integer',
        ]);

        $stockPotato = StockPotato::create($request->all());

        $qrData = "Name: {$stockPotato->stoneType->name}, Dimensions: {$stockPotato->length}x{$stockPotato->width}x{$stockPotato->height}";

        $qrCode = Builder::create()
            ->data($qrData)
            ->size(200)
            ->build();

        $fileName = 'qrcodes/' . $stockPotato->id . '.png';
        Storage::disk('public')->put($fileName, $qrCode->getString());

        $stockPotato->qr_code = $fileName;
        $stockPotato->save();

        return redirect()->route('stock_potatoes.index');
    }

    public function edit(StockPotato $stockPotato)
    {
        $stoneTypes = StoneType::all();
        return view('stock_potatoes.edit', compact('stockPotato', 'stoneTypes'));
    }

    public function update(Request $request, StockPotato $stockPotato)
    {
        $request->validate([
            'stone_type_id' => 'required|exists:stone_types,id',
            'length' => 'required|numeric',
            'width' => 'required|numeric',
            'height' => 'required|numeric',
            'quantity' => 'required|integer',
        ]);

        $stockPotato->update($request->all());

        $qrData = "Name: {$stockPotato->stoneType->name}, Dimensions: {$stockPotato->length}x{$stockPotato->width}x{$stockPotato->height}";

        $qrCode = Builder::create()
            ->data($qrData)
            ->size(200)
            ->build();

        $fileName = 'qrcodes/' . $stockPotato->id . '.png';
        Storage::disk('public')->put($fileName, $qrCode->getString());

        $stockPotato->qr_code = $fileName;
        $stockPotato->save();

        return redirect()->route('stock_potatoes.index');
    }

    public function destroy(StockPotato $stockPotato)
    {
        $stockPotato->delete();
        return redirect()->route('stock_potatoes.index');
    }

    public function downloadQrCode(StockPotato $stockPotato)
    {
        $filePath = storage_path('app/public/' . $stockPotato->qr_code);
        return response()->download($filePath);
    }
}
